// ============================================================
// CONEXUS OMS - MAIN APPLICATION
// ============================================================

const ConexusApp = {
    // Current state
    state: {
        currentPage: 'dashboard',
        selectedOrder: null,
        filters: {
            status: 'all',
            dateRange: 'all',
            supplier: 'all',
            alertPriority: 'all',
            alertType: 'all'
        },
        searchQuery: '',
        notifications: [],
        isLoading: false
    },

    // Initialize application
    init() {
        this.bindEvents();
        this.loadNotifications();
        this.renderCurrentPage();
        this.updateAlertBadges();
        this.setupKeyboardShortcuts();
        console.log('Conexus OMS initialized');
    },

    // Bind global events
    bindEvents() {
        // Navigation clicks
        document.querySelectorAll('.nav-item').forEach(item => {
            item.addEventListener('click', (e) => {
                e.preventDefault();
                const page = item.dataset.page;
                if (page) this.navigateTo(page);
            });
        });

        // Global search
        const searchInput = document.getElementById('globalSearch');
        if (searchInput) {
            searchInput.addEventListener('input', (e) => {
                this.handleGlobalSearch(e.target.value);
            });
            searchInput.addEventListener('keydown', (e) => {
                if (e.key === 'Enter') {
                    this.executeGlobalSearch(e.target.value);
                }
            });
        }

        // Modal close on backdrop click
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => {
            backdrop.addEventListener('click', (e) => {
                if (e.target === backdrop) {
                    this.closeModal(backdrop.id);
                }
            });
        });

        // Notification bell
        const notificationBtn = document.getElementById('notificationBtn');
        if (notificationBtn) {
            notificationBtn.addEventListener('click', () => this.toggleNotifications());
        }

        // Close dropdowns on outside click
        document.addEventListener('click', (e) => {
            if (!e.target.closest('.dropdown')) {
                document.querySelectorAll('.dropdown-menu.show').forEach(menu => {
                    menu.classList.remove('show');
                });
            }
        });
    },

    // Setup keyboard shortcuts
    setupKeyboardShortcuts() {
        document.addEventListener('keydown', (e) => {
            // Cmd/Ctrl + K for global search
            if ((e.metaKey || e.ctrlKey) && e.key === 'k') {
                e.preventDefault();
                document.getElementById('globalSearch')?.focus();
            }
            // Escape to close modals
            if (e.key === 'Escape') {
                document.querySelectorAll('.modal-backdrop.active').forEach(modal => {
                    this.closeModal(modal.id);
                });
            }
            // N for new order
            if (e.key === 'n' && !e.target.matches('input, textarea')) {
                e.preventDefault();
                this.openModal('createOrderModal');
            }
        });
    },

    // Navigate to page
    navigateTo(page) {
        this.state.currentPage = page;
        
        // Update nav active state
        document.querySelectorAll('.nav-item').forEach(item => {
            item.classList.toggle('active', item.dataset.page === page);
        });

        // Hide all pages
        document.querySelectorAll('.page-section').forEach(section => {
            section.classList.add('hidden');
        });

        // Show target page
        const targetPage = document.getElementById(`page-${page}`);
        if (targetPage) {
            targetPage.classList.remove('hidden');
        }

        // Update page title
        this.updatePageTitle(page);

        // Render page-specific content
        this.renderPageContent(page);
    },

    // Update page title
    updatePageTitle(page) {
        const titles = {
            'dashboard': 'Control Tower',
            'orders': 'All Orders',
            'alerts': 'Proactive Alerts',
            'order-intake': 'Order Intake',
            'order-desk': 'Order Desk',
            'forecasting': 'Demand Forecasting',
            'tracking': 'Shipment Tracking',
            'documents': 'Document Center',
            'analytics': 'Analytics & Reports',
            'co2': 'CO₂ Tracking',
            'settings': 'Settings'
        };
        const pageTitle = document.querySelector('.page-title');
        if (pageTitle) {
            pageTitle.textContent = titles[page] || 'Conexus OMS';
        }
    },

    // Render page content
    renderPageContent(page) {
        switch(page) {
            case 'dashboard':
                this.renderDashboard();
                break;
            case 'orders':
                this.renderOrdersList();
                break;
            case 'alerts':
                this.renderAlerts();
                break;
            case 'order-intake':
                this.renderOrderIntake();
                break;
            case 'order-desk':
                this.renderOrderDesk();
                break;
            case 'forecasting':
                this.renderForecasting();
                break;
            case 'analytics':
                this.renderAnalytics();
                break;
            case 'co2':
                this.renderCO2Tracking();
                break;
        }
    },

    // Render current page
    renderCurrentPage() {
        this.renderPageContent(this.state.currentPage);
    },

    // ==================== DASHBOARD ====================
    renderDashboard() {
        this.renderKPIs();
        this.renderPipeline();
        this.renderRecentOrders();
        this.renderRecentAlerts();
        this.renderQuickActions();
    },

    renderKPIs() {
        const kpiContainer = document.getElementById('kpiGrid');
        if (!kpiContainer) return;

        const criticalAlerts = ConexusHelpers.getCriticalAlertsCount();
        const openAlerts = ConexusHelpers.getOpenAlertsCount();
        const pendingOrders = ConexusData.orders.filter(o => o.dataQuality < 70).length;
        const otd = ConexusData.analytics.otd;
        const activeOrders = ConexusData.orders.filter(o => !['Delivered', 'Cancelled', 'Archived'].includes(o.status)).length;
        const arrivingThisWeek = ConexusData.orders.filter(o => {
            if (!o.eta) return false;
            const eta = new Date(o.eta);
            const today = new Date();
            const weekFromNow = new Date(today.getTime() + 7 * 24 * 60 * 60 * 1000);
            return eta >= today && eta <= weekFromNow;
        }).length;

        kpiContainer.innerHTML = `
            <div class="kpi-card" onclick="ConexusApp.navigateTo('alerts')">
                <div class="kpi-icon critical">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"/>
                        <line x1="12" y1="9" x2="12" y2="13"/><line x1="12" y1="17" x2="12.01" y2="17"/>
                    </svg>
                </div>
                <div class="kpi-content">
                    <div class="kpi-label">Critical Alerts</div>
                    <div class="kpi-value">${criticalAlerts}</div>
                    <div class="kpi-meta">
                        <span class="kpi-description">${openAlerts} total open alerts</span>
                    </div>
                </div>
            </div>
            <div class="kpi-card" onclick="ConexusApp.navigateTo('order-intake')">
                <div class="kpi-icon warning">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                        <polyline points="14 2 14 8 20 8"/><line x1="12" y1="18" x2="12" y2="12"/>
                        <line x1="9" y1="15" x2="15" y2="15"/>
                    </svg>
                </div>
                <div class="kpi-content">
                    <div class="kpi-label">Pending Validation</div>
                    <div class="kpi-value">${pendingOrders}</div>
                    <div class="kpi-meta">
                        <span class="kpi-description">Orders need data completion</span>
                    </div>
                </div>
            </div>
            <div class="kpi-card" onclick="ConexusApp.navigateTo('analytics')">
                <div class="kpi-icon success">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <polyline points="22 12 18 12 15 21 9 3 6 12 2 12"/>
                    </svg>
                </div>
                <div class="kpi-content">
                    <div class="kpi-label">On-Time Delivery</div>
                    <div class="kpi-value">${otd.current}%</div>
                    <div class="kpi-meta">
                        <span class="kpi-trend ${otd.trend}">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <polyline points="${otd.trend === 'up' ? '18 15 12 9 6 15' : '6 9 12 15 18 9'}"/>
                            </svg>
                            ${Math.abs(otd.current - otd.lastMonth)}%
                        </span>
                        <span class="kpi-description">vs last month</span>
                    </div>
                </div>
            </div>
            <div class="kpi-card" onclick="ConexusApp.navigateTo('orders')">
                <div class="kpi-icon info">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <rect x="2" y="3" width="20" height="14" rx="2" ry="2"/>
                        <line x1="8" y1="21" x2="16" y2="21"/><line x1="12" y1="17" x2="12" y2="21"/>
                    </svg>
                </div>
                <div class="kpi-content">
                    <div class="kpi-label">Active Orders</div>
                    <div class="kpi-value">${activeOrders}</div>
                    <div class="kpi-meta">
                        <span class="kpi-description">${arrivingThisWeek} arriving this week</span>
                    </div>
                </div>
            </div>
        `;
    },

    renderPipeline() {
        const pipelineContainer = document.getElementById('orderPipeline');
        if (!pipelineContainer) return;

        const counts = ConexusHelpers.getPipelineCounts();
        const stages = ['Total', 'Placed', 'Approved', 'In Production', 'Cargo Ready', 'Loaded', 'In Transit', 'Arrived', 'Delivered'];

        pipelineContainer.innerHTML = stages.map(stage => `
            <div class="pipeline-stage ${this.state.filters.status === stage ? 'active' : ''}" 
                 onclick="ConexusApp.filterByStatus('${stage}')">
                <div class="pipeline-count">${counts[stage] || 0}</div>
                <div class="pipeline-label">${stage}</div>
            </div>
        `).join('');
    },

    renderRecentOrders() {
        const container = document.getElementById('recentOrders');
        if (!container) return;

        const orders = ConexusData.orders.slice(0, 5);

        container.innerHTML = `
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Order #</th>
                        <th>Status</th>
                        <th>Data Quality</th>
                        <th>Supplier</th>
                        <th>ETA</th>
                        <th>Confidence</th>
                    </tr>
                </thead>
                <tbody>
                    ${orders.map(order => {
                        const supplier = ConexusHelpers.getSupplier(order.supplierId);
                        const quality = ConexusHelpers.getDataQualityLevel(order.dataQuality);
                        const confidence = ConexusHelpers.getConfidenceBadge(order.etaConfidence);
                        return `
                            <tr class="clickable" onclick="ConexusApp.openOrderDetail('${order.id}')">
                                <td><strong>${order.id}</strong></td>
                                <td><span class="badge badge-status" style="background: ${ConexusHelpers.getStatusColor(order.status)}20; color: ${ConexusHelpers.getStatusColor(order.status)}">${order.status}</span></td>
                                <td>
                                    <div class="data-quality">
                                        <div class="data-quality-bar">
                                            <div class="data-quality-fill ${quality.level.toLowerCase()}" style="width: ${order.dataQuality}%"></div>
                                        </div>
                                        <span class="data-quality-value" style="color: ${quality.color}">${order.dataQuality}%</span>
                                    </div>
                                </td>
                                <td>${supplier?.name || '-'}</td>
                                <td>${ConexusHelpers.formatDate(order.eta)}</td>
                                <td>
                                    <span class="confidence">
                                        <span class="confidence-dot ${order.etaConfidence?.toLowerCase()}"></span>
                                        ${order.etaConfidence || 'Unknown'}
                                    </span>
                                </td>
                            </tr>
                        `;
                    }).join('')}
                </tbody>
            </table>
        `;
    },

    renderRecentAlerts() {
        const container = document.getElementById('recentAlerts');
        if (!container) return;

        const alerts = ConexusData.alerts.filter(a => a.status === 'Open').slice(0, 3);

        container.innerHTML = alerts.map(alert => `
            <div class="alert-item" onclick="ConexusApp.openAlertDetail('${alert.id}')">
                <div class="alert-severity ${alert.priority.toLowerCase()}"></div>
                <div class="alert-content">
                    <div class="alert-header">
                        <span class="alert-title">${alert.title}</span>
                        <span class="alert-meta">
                            <span class="badge badge-${alert.priority === 'Critical' ? 'error' : alert.priority === 'High' ? 'warning' : 'info'}">${alert.priority}</span>
                        </span>
                    </div>
                    <div class="alert-description">${alert.description.substring(0, 100)}...</div>
                    <div class="alert-footer">
                        <div class="alert-tags">
                            <span class="badge badge-muted">${alert.affectedCount} order${alert.affectedCount > 1 ? 's' : ''}</span>
                            <span class="badge badge-muted">${alert.type}</span>
                        </div>
                        <span class="text-muted" style="font-size: 12px">${alert.createdAt}</span>
                    </div>
                </div>
            </div>
        `).join('');
    },

    renderQuickActions() {
        const container = document.getElementById('quickActions');
        if (!container) return;

        container.innerHTML = `
            <button class="btn btn-primary" onclick="ConexusApp.openModal('createOrderModal')">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <line x1="12" y1="5" x2="12" y2="19"/><line x1="5" y1="12" x2="19" y2="12"/>
                </svg>
                New Order
            </button>
            <button class="btn btn-secondary" onclick="ConexusApp.openModal('importOrdersModal')">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                    <polyline points="17 8 12 3 7 8"/><line x1="12" y1="3" x2="12" y2="15"/>
                </svg>
                Import Orders
            </button>
            <button class="btn btn-secondary" onclick="ConexusApp.navigateTo('analytics')">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <line x1="18" y1="20" x2="18" y2="10"/><line x1="12" y1="20" x2="12" y2="4"/>
                    <line x1="6" y1="20" x2="6" y2="14"/>
                </svg>
                View Reports
            </button>
        `;
    },

    // ==================== ORDERS LIST ====================
    renderOrdersList() {
        const container = document.getElementById('ordersTableContainer');
        if (!container) return;

        let orders = [...ConexusData.orders];

        // Apply filters
        if (this.state.filters.status !== 'all' && this.state.filters.status !== 'Total') {
            orders = orders.filter(o => o.status === this.state.filters.status);
        }

        if (this.state.searchQuery) {
            orders = ConexusHelpers.searchOrders(this.state.searchQuery);
        }

        container.innerHTML = `
            <div class="card">
                <div class="card-header">
                    <div>
                        <h3 class="card-title">Orders (${orders.length})</h3>
                        <p class="card-subtitle">Manage and track all purchase orders</p>
                    </div>
                    <div class="flex gap-sm">
                        <select class="form-select" style="width: 150px" onchange="ConexusApp.filterByStatus(this.value)">
                            <option value="all">All Statuses</option>
                            ${ConexusData.config.statuses.map(s => `<option value="${s}" ${this.state.filters.status === s ? 'selected' : ''}>${s}</option>`).join('')}
                        </select>
                        <button class="btn btn-primary" onclick="ConexusApp.openModal('createOrderModal')">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="16" height="16">
                                <line x1="12" y1="5" x2="12" y2="19"/><line x1="5" y1="12" x2="19" y2="12"/>
                            </svg>
                            New Order
                        </button>
                    </div>
                </div>
                <div class="table-container">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Order #</th>
                                <th>Status</th>
                                <th>Buyer</th>
                                <th>Supplier</th>
                                <th>Incoterm</th>
                                <th>Value</th>
                                <th>ETD</th>
                                <th>ETA</th>
                                <th>Data Quality</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${orders.map(order => {
                                const supplier = ConexusHelpers.getSupplier(order.supplierId);
                                const buyer = ConexusHelpers.getBuyer(order.buyerId);
                                const quality = ConexusHelpers.getDataQualityLevel(order.dataQuality);
                                return `
                                    <tr class="clickable">
                                        <td onclick="ConexusApp.openOrderDetail('${order.id}')"><strong>${order.id}</strong></td>
                                        <td><span class="badge badge-status" style="background: ${ConexusHelpers.getStatusColor(order.status)}20; color: ${ConexusHelpers.getStatusColor(order.status)}">${order.status}</span></td>
                                        <td>${buyer?.name || '-'}</td>
                                        <td>${supplier?.name || '-'}</td>
                                        <td><span class="badge badge-muted">${order.incoterm}</span></td>
                                        <td>${ConexusHelpers.formatCurrency(order.totalValue)}</td>
                                        <td>${ConexusHelpers.formatDate(order.etd)}</td>
                                        <td>
                                            <span class="confidence">
                                                <span class="confidence-dot ${order.etaConfidence?.toLowerCase()}"></span>
                                                ${ConexusHelpers.formatDate(order.eta)}
                                            </span>
                                        </td>
                                        <td>
                                            <div class="data-quality">
                                                <div class="data-quality-bar">
                                                    <div class="data-quality-fill ${quality.level.toLowerCase()}" style="width: ${order.dataQuality}%"></div>
                                                </div>
                                                <span class="data-quality-value" style="color: ${quality.color}">${order.dataQuality}%</span>
                                            </div>
                                        </td>
                                        <td>
                                            <div class="flex gap-sm">
                                                <button class="btn btn-icon btn-ghost sm" onclick="ConexusApp.openOrderDetail('${order.id}')" title="View">
                                                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"/><circle cx="12" cy="12" r="3"/></svg>
                                                </button>
                                                <button class="btn btn-icon btn-ghost sm" onclick="ConexusApp.editOrder('${order.id}')" title="Edit">
                                                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"/><path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"/></svg>
                                                </button>
                                                <button class="btn btn-icon btn-ghost sm" onclick="ConexusApp.messageOrder('${order.id}')" title="Message">
                                                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"/></svg>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                `;
                            }).join('')}
                        </tbody>
                    </table>
                </div>
            </div>
        `;
    },

    // ==================== ALERTS ====================
    renderAlerts() {
        const container = document.getElementById('alertsContainer');
        if (!container) return;

        let alerts = [...ConexusData.alerts];

        // Apply filters
        if (this.state.filters.alertPriority !== 'all') {
            alerts = alerts.filter(a => a.priority === this.state.filters.alertPriority);
        }
        if (this.state.filters.alertType !== 'all') {
            alerts = alerts.filter(a => a.type === this.state.filters.alertType);
        }

        const priorityCounts = {
            critical: ConexusData.alerts.filter(a => a.priority === 'Critical' && a.status === 'Open').length,
            high: ConexusData.alerts.filter(a => a.priority === 'High' && a.status === 'Open').length,
            medium: ConexusData.alerts.filter(a => a.priority === 'Medium' && a.status === 'Open').length,
            low: ConexusData.alerts.filter(a => a.priority === 'Low' && a.status === 'Open').length
        };

        container.innerHTML = `
            <div class="kpi-grid" style="margin-bottom: 24px">
                <div class="kpi-card" onclick="ConexusApp.filterAlerts('priority', 'Critical')">
                    <div class="kpi-icon critical">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <circle cx="12" cy="12" r="10"/><line x1="12" y1="8" x2="12" y2="12"/><line x1="12" y1="16" x2="12.01" y2="16"/>
                        </svg>
                    </div>
                    <div class="kpi-content">
                        <div class="kpi-label">Critical</div>
                        <div class="kpi-value">${priorityCounts.critical}</div>
                        <div class="kpi-meta"><span class="kpi-description">Immediate action required</span></div>
                    </div>
                </div>
                <div class="kpi-card" onclick="ConexusApp.filterAlerts('priority', 'High')">
                    <div class="kpi-icon warning">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"/>
                            <line x1="12" y1="9" x2="12" y2="13"/><line x1="12" y1="17" x2="12.01" y2="17"/>
                        </svg>
                    </div>
                    <div class="kpi-content">
                        <div class="kpi-label">High Priority</div>
                        <div class="kpi-value">${priorityCounts.high}</div>
                        <div class="kpi-meta"><span class="kpi-description">Address within 24 hours</span></div>
                    </div>
                </div>
                <div class="kpi-card" onclick="ConexusApp.filterAlerts('priority', 'Medium')">
                    <div class="kpi-icon info">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <circle cx="12" cy="12" r="10"/><line x1="12" y1="16" x2="12" y2="12"/><line x1="12" y1="8" x2="12.01" y2="8"/>
                        </svg>
                    </div>
                    <div class="kpi-content">
                        <div class="kpi-label">Medium</div>
                        <div class="kpi-value">${priorityCounts.medium}</div>
                        <div class="kpi-meta"><span class="kpi-description">Monitor and plan</span></div>
                    </div>
                </div>
                <div class="kpi-card" onclick="ConexusApp.filterAlerts('priority', 'Low')">
                    <div class="kpi-icon primary">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/>
                        </svg>
                    </div>
                    <div class="kpi-content">
                        <div class="kpi-label">Low Priority</div>
                        <div class="kpi-value">${priorityCounts.low}</div>
                        <div class="kpi-meta"><span class="kpi-description">Informational</span></div>
                    </div>
                </div>
            </div>

            <div class="card">
                <div class="card-header">
                    <div>
                        <h3 class="card-title">Proactive Alerts</h3>
                        <p class="card-subtitle">AI-powered alerts and recommendations</p>
                    </div>
                    <div class="flex gap-sm">
                        <select class="form-select" style="width: 140px" onchange="ConexusApp.filterAlerts('priority', this.value)">
                            <option value="all">All Priorities</option>
                            ${ConexusData.config.alertPriorities.map(p => `<option value="${p}">${p}</option>`).join('')}
                        </select>
                        <select class="form-select" style="width: 160px" onchange="ConexusApp.filterAlerts('type', this.value)">
                            <option value="all">All Types</option>
                            ${ConexusData.config.exceptionTypes.map(t => `<option value="${t}">${t}</option>`).join('')}
                        </select>
                    </div>
                </div>
                <div class="card-body" style="padding: 16px">
                    ${alerts.length === 0 ? `
                        <div class="empty-state">
                            <svg class="empty-state-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/>
                            </svg>
                            <h3 class="empty-state-title">No alerts</h3>
                            <p class="empty-state-description">All systems are running smoothly</p>
                        </div>
                    ` : alerts.map(alert => `
                        <div class="alert-item">
                            <div class="alert-severity ${alert.priority.toLowerCase()}"></div>
                            <div class="alert-content">
                                <div class="alert-header">
                                    <span class="alert-title">${alert.title}</span>
                                    <span class="alert-meta">
                                        <span class="badge badge-${alert.priority === 'Critical' ? 'error' : alert.priority === 'High' ? 'warning' : alert.priority === 'Medium' ? 'info' : 'muted'}">${alert.priority}</span>
                                        <span>${alert.createdAt}</span>
                                    </span>
                                </div>
                                <div class="alert-description">${alert.description}</div>
                                ${alert.recommendation ? `
                                    <div style="background: var(--accent-light); padding: 12px; border-radius: 6px; margin: 12px 0; font-size: 13px;">
                                        <strong style="color: var(--primary)">💡 Recommendation:</strong> ${alert.recommendation}
                                    </div>
                                ` : ''}
                                <div class="alert-footer">
                                    <div class="alert-tags">
                                        <span class="badge badge-muted">${alert.type}</span>
                                        <span class="badge badge-muted">${alert.affectedCount} order${alert.affectedCount > 1 ? 's' : ''}</span>
                                        ${alert.assignedTo ? `<span class="badge badge-primary">${alert.assignedTo}</span>` : '<span class="badge badge-warning">Unassigned</span>'}
                                        <span class="badge ${alert.status === 'Open' ? 'badge-error' : 'badge-success'}">${alert.status}</span>
                                    </div>
                                    <div class="alert-actions">
                                        ${alert.actions.map(action => `
                                            <button class="btn btn-sm btn-secondary" onclick="ConexusApp.handleAlertAction('${alert.id}', '${action}')">${action}</button>
                                        `).join('')}
                                        <button class="btn btn-sm btn-primary" onclick="ConexusApp.acknowledgeAlert('${alert.id}')">
                                            ${alert.status === 'Acknowledged' ? 'Resolve' : 'Acknowledge'}
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    `).join('')}
                </div>
            </div>
        `;
    },

    // ==================== ORDER INTAKE ====================
    renderOrderIntake() {
        const container = document.getElementById('orderIntakeContainer');
        if (!container) return;

        const validOrders = ConexusData.orders.filter(o => o.dataQuality >= 90);
        const missingFields = ConexusData.orders.filter(o => o.dataQuality >= 70 && o.dataQuality < 90);
        const lowQuality = ConexusData.orders.filter(o => o.dataQuality < 70);

        container.innerHTML = `
            <div class="grid-2" style="margin-bottom: 24px">
                <div class="card">
                    <div class="card-body">
                        <div class="upload-zone" id="uploadZone">
                            <svg class="upload-zone-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                                <polyline points="17 8 12 3 7 8"/><line x1="12" y1="3" x2="12" y2="15"/>
                            </svg>
                            <h3 class="upload-zone-title">Drop files here or click to upload</h3>
                            <p class="upload-zone-subtitle">Support for Excel, CSV, EDI, and PDF formats</p>
                            <div class="upload-formats">
                                <span class="badge badge-success">Excel</span>
                                <span class="badge badge-success">CSV</span>
                                <span class="badge badge-info">EDI</span>
                                <span class="badge badge-warning">PDF</span>
                            </div>
                        </div>
                        <input type="file" id="fileInput" hidden multiple accept=".xlsx,.xls,.csv,.edi,.pdf">
                    </div>
                </div>
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">Validation Summary</h3>
                    </div>
                    <div class="card-body">
                        <div class="flex items-center justify-between" style="padding: 12px 0; border-bottom: 1px solid var(--border)">
                            <div class="flex items-center gap-sm">
                                <div style="width: 12px; height: 12px; border-radius: 50%; background: var(--success)"></div>
                                <span>Valid Orders</span>
                            </div>
                            <strong>${validOrders.length}</strong>
                        </div>
                        <div class="flex items-center justify-between" style="padding: 12px 0; border-bottom: 1px solid var(--border)">
                            <div class="flex items-center gap-sm">
                                <div style="width: 12px; height: 12px; border-radius: 50%; background: var(--warning)"></div>
                                <span>Missing Fields</span>
                            </div>
                            <strong>${missingFields.length}</strong>
                        </div>
                        <div class="flex items-center justify-between" style="padding: 12px 0; border-bottom: 1px solid var(--border)">
                            <div class="flex items-center gap-sm">
                                <div style="width: 12px; height: 12px; border-radius: 50%; background: var(--error)"></div>
                                <span>Low Quality / Issues</span>
                            </div>
                            <strong>${lowQuality.length}</strong>
                        </div>
                        <div class="flex items-center justify-between" style="padding: 12px 0">
                            <div class="flex items-center gap-sm">
                                <div style="width: 12px; height: 12px; border-radius: 50%; background: var(--info)"></div>
                                <span>Total Orders</span>
                            </div>
                            <strong>${ConexusData.orders.length}</strong>
                        </div>
                    </div>
                </div>
            </div>

            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">Orders Pending Validation</h3>
                    <button class="btn btn-primary btn-sm" onclick="ConexusApp.validateAllOrders()">Validate All</button>
                </div>
                <div class="table-container">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Order #</th>
                                <th>Source</th>
                                <th>Supplier</th>
                                <th>Data Quality</th>
                                <th>Issues</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${[...missingFields, ...lowQuality].map(order => {
                                const supplier = ConexusHelpers.getSupplier(order.supplierId);
                                const quality = ConexusHelpers.getDataQualityLevel(order.dataQuality);
                                const issues = [];
                                if (!order.eta) issues.push('Missing ETA');
                                if (!order.cargoReadyDate) issues.push('Missing Cargo Ready Date');
                                if (order.documents.filter(d => d.status === 'Missing').length > 0) issues.push('Missing Documents');
                                if (!order.containerRef && order.mode.includes('Sea')) issues.push('Missing Container Ref');
                                
                                return `
                                    <tr>
                                        <td><strong>${order.id}</strong></td>
                                        <td><span class="badge badge-muted">Manual Entry</span></td>
                                        <td>${supplier?.name || '-'}</td>
                                        <td>
                                            <div class="data-quality">
                                                <div class="data-quality-bar">
                                                    <div class="data-quality-fill ${quality.level.toLowerCase()}" style="width: ${order.dataQuality}%"></div>
                                                </div>
                                                <span class="data-quality-value" style="color: ${quality.color}">${order.dataQuality}%</span>
                                            </div>
                                        </td>
                                        <td style="max-width: 200px">
                                            <span class="text-muted" style="font-size: 12px">${issues.join(', ') || 'Review recommended'}</span>
                                        </td>
                                        <td>
                                            <div class="flex gap-sm">
                                                <button class="btn btn-sm btn-primary" onclick="ConexusApp.completeOrderData('${order.id}')">Complete</button>
                                                <button class="btn btn-sm btn-secondary" onclick="ConexusApp.openOrderDetail('${order.id}')">View</button>
                                            </div>
                                        </td>
                                    </tr>
                                `;
                            }).join('')}
                        </tbody>
                    </table>
                </div>
            </div>
        `;

        // Setup upload zone
        this.setupUploadZone();
    },

    setupUploadZone() {
        const uploadZone = document.getElementById('uploadZone');
        const fileInput = document.getElementById('fileInput');

        if (!uploadZone || !fileInput) return;

        uploadZone.addEventListener('click', () => fileInput.click());
        uploadZone.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadZone.classList.add('dragover');
        });
        uploadZone.addEventListener('dragleave', () => {
            uploadZone.classList.remove('dragover');
        });
        uploadZone.addEventListener('drop', (e) => {
            e.preventDefault();
            uploadZone.classList.remove('dragover');
            const files = e.dataTransfer.files;
            this.handleFileUpload(files);
        });
        fileInput.addEventListener('change', (e) => {
            this.handleFileUpload(e.target.files);
        });
    },

    handleFileUpload(files) {
        if (files.length === 0) return;
        
        // Simulate file processing
        this.showToast(`Processing ${files.length} file(s)...`, 'info');
        
        setTimeout(() => {
            this.showToast(`Successfully imported ${files.length} file(s)`, 'success');
            // In real implementation, would parse files and add to orders
        }, 1500);
    },

    // ==================== ORDER DESK ====================
    renderOrderDesk() {
        const container = document.getElementById('orderDeskContainer');
        if (!container) return;

        const orders = ConexusData.orders.slice(0, 10);

        container.innerHTML = `
            <div style="display: flex; gap: 20px; height: calc(100vh - 180px)">
                <div style="width: 350px; flex-shrink: 0; display: flex; flex-direction: column">
                    <div class="card" style="flex: 1; display: flex; flex-direction: column">
                        <div class="card-header">
                            <input type="text" class="form-input" placeholder="Search orders..." style="width: 100%" id="orderDeskSearch" oninput="ConexusApp.filterOrderDesk(this.value)">
                        </div>
                        <div style="flex: 1; overflow-y: auto" id="orderDeskList">
                            ${orders.map((order, i) => {
                                const supplier = ConexusHelpers.getSupplier(order.supplierId);
                                return `
                                    <div class="order-desk-item ${i === 0 ? 'active' : ''}" onclick="ConexusApp.selectOrderDesk('${order.id}')" data-order-id="${order.id}">
                                        <div style="display: flex; justify-content: space-between; margin-bottom: 4px">
                                            <strong>${order.id}</strong>
                                            <span class="badge badge-status" style="background: ${ConexusHelpers.getStatusColor(order.status)}20; color: ${ConexusHelpers.getStatusColor(order.status)}; font-size: 10px">${order.status}</span>
                                        </div>
                                        <div style="font-size: 12px; color: var(--text-muted)">${supplier?.name || '-'}</div>
                                        <div style="font-size: 11px; color: var(--text-light); margin-top: 4px">ETA: ${ConexusHelpers.formatDate(order.eta)}</div>
                                    </div>
                                `;
                            }).join('')}
                        </div>
                    </div>
                </div>
                <div style="flex: 1; min-width: 0">
                    <div class="card" style="height: 100%; display: flex; flex-direction: column">
                        <div id="orderDeskDetail" style="flex: 1; overflow-y: auto">
                            ${this.renderOrderDeskDetail(orders[0])}
                        </div>
                    </div>
                </div>
            </div>
        `;

        // Add styles for order desk items
        const style = document.createElement('style');
        style.textContent = `
            .order-desk-item {
                padding: 12px 16px;
                border-bottom: 1px solid var(--border);
                cursor: pointer;
                transition: background 0.15s;
            }
            .order-desk-item:hover {
                background: var(--bg-light);
            }
            .order-desk-item.active {
                background: var(--accent-light);
                border-left: 3px solid var(--accent);
            }
        `;
        document.head.appendChild(style);
    },

    renderOrderDeskDetail(order) {
        if (!order) return '<div class="empty-state"><p>Select an order to view details</p></div>';

        const supplier = ConexusHelpers.getSupplier(order.supplierId);
        const buyer = ConexusHelpers.getBuyer(order.buyerId);
        const forwarder = ConexusHelpers.getForwarder(order.forwarderId);

        return `
            <div class="card-header" style="border-bottom: 1px solid var(--border)">
                <div>
                    <h3 class="card-title">${order.id}</h3>
                    <p class="card-subtitle">${supplier?.name} → ${buyer?.name}</p>
                </div>
                <div class="flex gap-sm">
                    ${order.co2Emissions ? `<span class="badge badge-success">🌱 ${order.co2Emissions} kg CO₂</span>` : ''}
                    <button class="btn btn-sm btn-secondary" onclick="ConexusApp.messageOrder('${order.id}')">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="14" height="14">
                            <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"/>
                        </svg>
                        Message
                    </button>
                    <button class="btn btn-sm btn-primary" onclick="ConexusApp.openOrderDetail('${order.id}')">Full Details</button>
                </div>
            </div>
            <div style="padding: 20px">
                <h4 style="font-size: 14px; font-weight: 600; margin-bottom: 16px; color: var(--text)">Order Timeline</h4>
                <div class="timeline">
                    ${order.milestones.map(m => `
                        <div class="timeline-item">
                            <div class="timeline-marker ${m.completed ? 'completed' : m.date ? 'current' : 'pending'}">
                                ${m.completed ? '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3"><polyline points="20 6 9 17 4 12"/></svg>' : ''}
                            </div>
                            <div class="timeline-content">
                                <div class="timeline-header">
                                    <span class="timeline-title">${m.name}</span>
                                    <span class="timeline-date">${m.date ? ConexusHelpers.formatDate(m.date) : 'Pending'}</span>
                                </div>
                                ${m.note ? `<div class="timeline-note">${m.note}</div>` : ''}
                            </div>
                        </div>
                    `).join('')}
                </div>

                <h4 style="font-size: 14px; font-weight: 600; margin: 24px 0 16px; color: var(--text)">Shipment Details</h4>
                <div class="grid-2" style="gap: 12px">
                    <div style="padding: 12px; background: var(--bg-light); border-radius: 6px">
                        <div style="font-size: 11px; color: var(--text-muted); margin-bottom: 4px">Container</div>
                        <div style="font-weight: 600">${order.containerRef || '-'}</div>
                    </div>
                    <div style="padding: 12px; background: var(--bg-light); border-radius: 6px">
                        <div style="font-size: 11px; color: var(--text-muted); margin-bottom: 4px">Vessel / Flight</div>
                        <div style="font-weight: 600">${order.vesselName || '-'}</div>
                    </div>
                    <div style="padding: 12px; background: var(--bg-light); border-radius: 6px">
                        <div style="font-size: 11px; color: var(--text-muted); margin-bottom: 4px">MBL / AWB</div>
                        <div style="font-weight: 600">${order.mblNumber || '-'}</div>
                    </div>
                    <div style="padding: 12px; background: var(--bg-light); border-radius: 6px">
                        <div style="font-size: 11px; color: var(--text-muted); margin-bottom: 4px">Forwarder</div>
                        <div style="font-weight: 600">${forwarder?.name || '-'}</div>
                    </div>
                </div>

                ${order.vesselName ? `
                    <div style="margin-top: 16px">
                        <button class="btn btn-secondary" onclick="ConexusApp.trackVessel('${order.id}')">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="16" height="16">
                                <circle cx="12" cy="12" r="10"/><polygon points="16.24 7.76 14.12 14.12 7.76 16.24 9.88 9.88 16.24 7.76"/>
                            </svg>
                            Track Vessel
                        </button>
                    </div>
                ` : ''}
            </div>
        `;
    },

    selectOrderDesk(orderId) {
        const order = ConexusHelpers.getOrder(orderId);
        if (!order) return;

        // Update active state
        document.querySelectorAll('.order-desk-item').forEach(item => {
            item.classList.toggle('active', item.dataset.orderId === orderId);
        });

        // Render detail
        const detailContainer = document.getElementById('orderDeskDetail');
        if (detailContainer) {
            detailContainer.innerHTML = this.renderOrderDeskDetail(order);
        }
    },

    // ==================== FORECASTING ====================
    renderForecasting() {
        const container = document.getElementById('forecastingContainer');
        if (!container) return;

        const forecast = ConexusData.forecasting;
        const currentMonth = forecast.forecast[0];
        const nextMonth = forecast.forecast[1];

        container.innerHTML = `
            <div class="kpi-grid" style="margin-bottom: 24px">
                <div class="kpi-card">
                    <div class="kpi-icon info">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <line x1="12" y1="20" x2="12" y2="10"/><line x1="18" y1="20" x2="18" y2="4"/>
                            <line x1="6" y1="20" x2="6" y2="16"/>
                        </svg>
                    </div>
                    <div class="kpi-content">
                        <div class="kpi-label">Forecasted Orders (Jan)</div>
                        <div class="kpi-value">${currentMonth.orders}</div>
                        <div class="kpi-meta">
                            <span class="badge badge-success">${currentMonth.confidence}% confidence</span>
                        </div>
                    </div>
                </div>
                <div class="kpi-card">
                    <div class="kpi-icon success">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <line x1="12" y1="1" x2="12" y2="23"/><path d="M17 5H9.5a3.5 3.5 0 0 0 0 7h5a3.5 3.5 0 0 1 0 7H6"/>
                        </svg>
                    </div>
                    <div class="kpi-content">
                        <div class="kpi-label">Projected Value (Jan)</div>
                        <div class="kpi-value">${ConexusHelpers.formatCurrency(currentMonth.value)}</div>
                        <div class="kpi-meta">
                            <span class="kpi-trend up">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="18 15 12 9 6 15"/></svg>
                                12%
                            </span>
                            <span class="kpi-description">vs Dec 2024</span>
                        </div>
                    </div>
                </div>
                <div class="kpi-card">
                    <div class="kpi-icon warning">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                            <line x1="3" y1="9" x2="21" y2="9"/><line x1="9" y1="21" x2="9" y2="9"/>
                        </svg>
                    </div>
                    <div class="kpi-content">
                        <div class="kpi-label">Capacity Utilization</div>
                        <div class="kpi-value">${forecast.capacity.currentMonthUtilization}%</div>
                        <div class="kpi-meta">
                            <span class="kpi-description">of ${forecast.capacity.currentMonthCapacity} orders/month</span>
                        </div>
                    </div>
                </div>
                <div class="kpi-card">
                    <div class="kpi-icon primary">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"/>
                        </svg>
                    </div>
                    <div class="kpi-content">
                        <div class="kpi-label">Warehouse Space</div>
                        <div class="kpi-value">${forecast.capacity.warehouseSpace.current}%</div>
                        <div class="kpi-meta">
                            <span class="kpi-description">Projected: ${forecast.capacity.warehouseSpace.projected}%</span>
                        </div>
                    </div>
                </div>
            </div>

            <div class="grid-2" style="margin-bottom: 24px">
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">6-Month Order Forecast</h3>
                    </div>
                    <div class="card-body">
                        <div style="display: flex; align-items: flex-end; gap: 16px; height: 200px; padding: 20px 0">
                            ${forecast.forecast.map(f => {
                                const height = (f.orders / 100) * 180;
                                return `
                                    <div style="flex: 1; display: flex; flex-direction: column; align-items: center">
                                        <div style="font-size: 12px; font-weight: 600; margin-bottom: 8px">${f.orders}</div>
                                        <div style="width: 100%; height: ${height}px; background: linear-gradient(to top, var(--primary), var(--info)); border-radius: 4px 4px 0 0; position: relative">
                                            <div style="position: absolute; top: -4px; left: 50%; transform: translateX(-50%); width: 8px; height: 8px; background: var(--accent); border-radius: 50%"></div>
                                        </div>
                                        <div style="font-size: 11px; color: var(--text-muted); margin-top: 8px">${f.month.split('-')[1]}</div>
                                    </div>
                                `;
                            }).join('')}
                        </div>
                        <div style="text-align: center; font-size: 12px; color: var(--text-muted); border-top: 1px solid var(--border); padding-top: 12px">
                            Confidence decreases over time: ${forecast.forecast.map(f => `${f.confidence}%`).join(' → ')}
                        </div>
                    </div>
                </div>

                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">Seasonality Index</h3>
                    </div>
                    <div class="card-body">
                        <div style="display: flex; gap: 8px; height: 200px; align-items: flex-end; padding: 20px 0">
                            ${forecast.seasonality.map(s => {
                                const height = (s.index / 130) * 180;
                                const isHigh = s.index >= 115;
                                const isLow = s.index <= 90;
                                return `
                                    <div style="flex: 1; display: flex; flex-direction: column; align-items: center">
                                        <div style="font-size: 10px; font-weight: 500; margin-bottom: 4px; color: ${isHigh ? 'var(--success)' : isLow ? 'var(--error)' : 'var(--text-muted)'}">${s.index}</div>
                                        <div style="width: 100%; height: ${height}px; background: ${isHigh ? 'var(--success)' : isLow ? 'var(--error)' : 'var(--info)'}; border-radius: 2px; opacity: 0.7"></div>
                                        <div style="font-size: 9px; color: var(--text-muted); margin-top: 4px">${s.month.substring(0, 3)}</div>
                                    </div>
                                `;
                            }).join('')}
                        </div>
                        <div style="display: flex; justify-content: center; gap: 20px; font-size: 11px; margin-top: 12px">
                            <span><span style="display: inline-block; width: 10px; height: 10px; background: var(--success); border-radius: 2px; margin-right: 4px"></span>Peak Season (115+)</span>
                            <span><span style="display: inline-block; width: 10px; height: 10px; background: var(--error); border-radius: 2px; margin-right: 4px"></span>Low Season (≤90)</span>
                        </div>
                    </div>
                </div>
            </div>

            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">Supplier Forecast (Q1 2025)</h3>
                    <p class="card-subtitle">Projected orders and trends by supplier</p>
                </div>
                <div class="table-container">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Supplier</th>
                                <th>Q1 Orders</th>
                                <th>Q1 Value</th>
                                <th>Trend</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${forecast.bySupplier.map(s => `
                                <tr>
                                    <td><strong>${s.name}</strong></td>
                                    <td>${s.q1Orders}</td>
                                    <td>${ConexusHelpers.formatCurrency(s.q1Value)}</td>
                                    <td>
                                        <span class="badge ${s.trend === 'up' ? 'badge-success' : s.trend === 'down' ? 'badge-error' : 'badge-muted'}">
                                            ${s.trend === 'up' ? '↑ Growing' : s.trend === 'down' ? '↓ Declining' : '→ Stable'}
                                        </span>
                                    </td>
                                    <td>
                                        <button class="btn btn-sm btn-secondary" onclick="ConexusApp.viewSupplierForecast('${s.supplierId}')">View Details</button>
                                    </td>
                                </tr>
                            `).join('')}
                        </tbody>
                    </table>
                </div>
            </div>
        `;
    },

    // ==================== ANALYTICS ====================
    renderAnalytics() {
        const container = document.getElementById('analyticsContainer');
        if (!container) return;

        const analytics = ConexusData.analytics;

        container.innerHTML = `
            <div class="kpi-grid" style="margin-bottom: 24px">
                <div class="kpi-card">
                    <div class="kpi-icon success">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/>
                        </svg>
                    </div>
                    <div class="kpi-content">
                        <div class="kpi-label">On-Time Delivery Rate</div>
                        <div class="kpi-value">${analytics.otd.current}%</div>
                        <div class="kpi-meta">
                            <span class="kpi-trend up">↑ ${analytics.otd.current - analytics.otd.lastMonth}%</span>
                            <span class="kpi-description">Target: ${analytics.otd.target}%</span>
                        </div>
                    </div>
                </div>
                <div class="kpi-card">
                    <div class="kpi-icon info">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <circle cx="12" cy="12" r="10"/><polyline points="12 6 12 12 16 14"/>
                        </svg>
                    </div>
                    <div class="kpi-content">
                        <div class="kpi-label">Avg Lead Time</div>
                        <div class="kpi-value">${analytics.performance.avgLeadTime} days</div>
                        <div class="kpi-meta">
                            <span class="kpi-description">Order to delivery</span>
                        </div>
                    </div>
                </div>
                <div class="kpi-card">
                    <div class="kpi-icon warning">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                            <polyline points="14 2 14 8 20 8"/><line x1="16" y1="13" x2="8" y2="13"/>
                            <line x1="16" y1="17" x2="8" y2="17"/>
                        </svg>
                    </div>
                    <div class="kpi-content">
                        <div class="kpi-label">Document Compliance</div>
                        <div class="kpi-value">${analytics.performance.documentComplianceRate}%</div>
                        <div class="kpi-meta">
                            <span class="kpi-description">Complete on first submission</span>
                        </div>
                    </div>
                </div>
                <div class="kpi-card">
                    <div class="kpi-icon primary">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"/>
                        </svg>
                    </div>
                    <div class="kpi-content">
                        <div class="kpi-label">Exception Rate</div>
                        <div class="kpi-value">${analytics.performance.exceptionRate}%</div>
                        <div class="kpi-meta">
                            <span class="kpi-description">Orders with exceptions</span>
                        </div>
                    </div>
                </div>
            </div>

            <div class="grid-2" style="margin-bottom: 24px">
                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">Supplier OTD Performance</h3>
                    </div>
                    <div class="card-body">
                        ${analytics.otd.bySupplier.map(s => `
                            <div style="display: flex; align-items: center; gap: 12px; padding: 8px 0; border-bottom: 1px solid var(--border)">
                                <div style="flex: 1; font-size: 13px">${s.name}</div>
                                <div style="width: 120px">
                                    <div style="height: 8px; background: var(--bg-muted); border-radius: 4px; overflow: hidden">
                                        <div style="height: 100%; width: ${s.rate}%; background: ${s.rate >= 90 ? 'var(--success)' : s.rate >= 80 ? 'var(--warning)' : 'var(--error)'}"></div>
                                    </div>
                                </div>
                                <div style="width: 45px; text-align: right; font-weight: 600; color: ${s.rate >= 90 ? 'var(--success)' : s.rate >= 80 ? 'var(--warning)' : 'var(--error)'}">${s.rate}%</div>
                            </div>
                        `).join('')}
                    </div>
                </div>

                <div class="card">
                    <div class="card-header">
                        <h3 class="card-title">CO₂ Emissions by Mode</h3>
                    </div>
                    <div class="card-body">
                        <div style="display: flex; justify-content: center; gap: 40px; padding: 20px 0">
                            <div style="text-align: center">
                                <div style="width: 100px; height: 100px; border-radius: 50%; background: conic-gradient(var(--info) 0% ${analytics.co2.seaFreight}%, var(--warning) ${analytics.co2.seaFreight}% ${analytics.co2.seaFreight + analytics.co2.airFreight}%, var(--success) ${analytics.co2.seaFreight + analytics.co2.airFreight}% 100%); display: flex; align-items: center; justify-content: center">
                                    <div style="width: 70px; height: 70px; background: white; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-weight: 700">${analytics.co2.totalEmissions}t</div>
                                </div>
                            </div>
                            <div style="display: flex; flex-direction: column; justify-content: center; gap: 12px">
                                <div style="display: flex; align-items: center; gap: 8px; font-size: 13px">
                                    <div style="width: 12px; height: 12px; background: var(--info); border-radius: 2px"></div>
                                    Sea Freight: ${analytics.co2.seaFreight}%
                                </div>
                                <div style="display: flex; align-items: center; gap: 8px; font-size: 13px">
                                    <div style="width: 12px; height: 12px; background: var(--warning); border-radius: 2px"></div>
                                    Air Freight: ${analytics.co2.airFreight}%
                                </div>
                                <div style="display: flex; align-items: center; gap: 8px; font-size: 13px">
                                    <div style="width: 12px; height: 12px; background: var(--success); border-radius: 2px"></div>
                                    Road/Other: ${analytics.co2.road}%
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">Monthly CO₂ Emissions Trend</h3>
                </div>
                <div class="card-body">
                    <div style="display: flex; align-items: flex-end; gap: 24px; height: 200px; padding: 20px 40px">
                        ${analytics.co2.monthlyTrend.map(m => {
                            const height = (m.emissions / 200) * 180;
                            return `
                                <div style="flex: 1; display: flex; flex-direction: column; align-items: center">
                                    <div style="font-size: 12px; font-weight: 600; margin-bottom: 8px">${m.emissions}t</div>
                                    <div style="width: 100%; height: ${height}px; background: var(--success); border-radius: 4px; opacity: 0.8"></div>
                                    <div style="font-size: 12px; color: var(--text-muted); margin-top: 8px">${m.month}</div>
                                </div>
                            `;
                        }).join('')}
                    </div>
                </div>
            </div>
        `;
    },

    // ==================== CO2 TRACKING ====================
    renderCO2Tracking() {
        const container = document.getElementById('co2Container');
        if (!container) return;

        const analytics = ConexusData.analytics;
        const ordersWithCO2 = ConexusData.orders.filter(o => o.co2Emissions);

        container.innerHTML = `
            <div class="kpi-grid" style="margin-bottom: 24px">
                <div class="kpi-card">
                    <div class="kpi-icon success">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"/>
                        </svg>
                    </div>
                    <div class="kpi-content">
                        <div class="kpi-label">Total CO₂ Emissions</div>
                        <div class="kpi-value">${analytics.co2.totalEmissions} t</div>
                        <div class="kpi-meta">
                            <span class="kpi-description">This month</span>
                        </div>
                    </div>
                </div>
                <div class="kpi-card">
                    <div class="kpi-icon info">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <rect x="1" y="3" width="15" height="13"/><polygon points="16 8 20 8 23 11 23 16 16 16 16 8"/>
                            <circle cx="5.5" cy="18.5" r="2.5"/><circle cx="18.5" cy="18.5" r="2.5"/>
                        </svg>
                    </div>
                    <div class="kpi-content">
                        <div class="kpi-label">Avg per Shipment</div>
                        <div class="kpi-value">${analytics.co2.avgPerShipment} kg</div>
                        <div class="kpi-meta">
                            <span class="kpi-description">GLEC compliant calculation</span>
                        </div>
                    </div>
                </div>
                <div class="kpi-card">
                    <div class="kpi-icon warning">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M22 12h-4l-3 9L9 3l-3 9H2"/>
                        </svg>
                    </div>
                    <div class="kpi-content">
                        <div class="kpi-label">Sea Freight %</div>
                        <div class="kpi-value">${analytics.co2.seaFreight}%</div>
                        <div class="kpi-meta">
                            <span class="kpi-description">Lowest carbon option</span>
                        </div>
                    </div>
                </div>
                <div class="kpi-card">
                    <div class="kpi-icon primary">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M12 2L2 7l10 5 10-5-10-5z"/><path d="M2 17l10 5 10-5"/>
                            <path d="M2 12l10 5 10-5"/>
                        </svg>
                    </div>
                    <div class="kpi-content">
                        <div class="kpi-label">Air Freight %</div>
                        <div class="kpi-value">${analytics.co2.airFreight}%</div>
                        <div class="kpi-meta">
                            <span class="kpi-description">Higher carbon impact</span>
                        </div>
                    </div>
                </div>
            </div>

            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">CO₂ Emissions by Order</h3>
                    <button class="btn btn-secondary btn-sm" onclick="ConexusApp.exportCO2Report()">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="14" height="14">
                            <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                            <polyline points="7 10 12 15 17 10"/><line x1="12" y1="15" x2="12" y2="3"/>
                        </svg>
                        Export Report
                    </button>
                </div>
                <div class="table-container">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Order #</th>
                                <th>Mode</th>
                                <th>Route</th>
                                <th>CO₂ Emissions</th>
                                <th>Status</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${ordersWithCO2.map(order => `
                                <tr class="clickable" onclick="ConexusApp.openOrderDetail('${order.id}')">
                                    <td><strong>${order.id}</strong></td>
                                    <td><span class="badge badge-muted">${order.mode}</span></td>
                                    <td>${order.portOfLoading} → ${order.portOfDischarge}</td>
                                    <td>
                                        <span class="badge badge-success">🌱 ${order.co2Emissions} kg</span>
                                    </td>
                                    <td><span class="badge badge-status" style="background: ${ConexusHelpers.getStatusColor(order.status)}20; color: ${ConexusHelpers.getStatusColor(order.status)}">${order.status}</span></td>
                                </tr>
                            `).join('')}
                        </tbody>
                    </table>
                </div>
            </div>
        `;
    },

    // ==================== MODALS ====================
    openModal(modalId) {
        const modal = document.getElementById(modalId);
        if (modal) {
            modal.classList.add('active');
            document.body.style.overflow = 'hidden';
        }
    },

    closeModal(modalId) {
        const modal = document.getElementById(modalId);
        if (modal) {
            modal.classList.remove('active');
            document.body.style.overflow = '';
        }
    },

    // Open order detail modal
    openOrderDetail(orderId) {
        const order = ConexusHelpers.getOrder(orderId);
        if (!order) return;

        this.state.selectedOrder = order;
        const modal = document.getElementById('orderDetailModal');
        const content = document.getElementById('orderDetailContent');

        if (!modal || !content) return;

        const supplier = ConexusHelpers.getSupplier(order.supplierId);
        const buyer = ConexusHelpers.getBuyer(order.buyerId);
        const originAgent = ConexusHelpers.getAgent(order.originAgentId, 'origin');
        const destAgent = ConexusHelpers.getAgent(order.destAgentId, 'destination');
        const forwarder = ConexusHelpers.getForwarder(order.forwarderId);
        const quality = ConexusHelpers.getDataQualityLevel(order.dataQuality);

        content.innerHTML = `
            <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 24px">
                <div>
                    <h2 style="font-size: 24px; font-weight: 700; margin-bottom: 4px">${order.id}</h2>
                    <p style="color: var(--text-muted)">${supplier?.name} → ${buyer?.name}</p>
                </div>
                <div class="flex gap-sm">
                    <span class="badge badge-status" style="background: ${ConexusHelpers.getStatusColor(order.status)}20; color: ${ConexusHelpers.getStatusColor(order.status)}">${order.status}</span>
                    <div class="data-quality">
                        <span class="data-quality-value" style="color: ${quality.color}">${order.dataQuality}% Quality</span>
                    </div>
                    ${order.co2Emissions ? `<span class="badge badge-success">🌱 ${order.co2Emissions} kg CO₂</span>` : ''}
                </div>
            </div>

            <div class="tabs" id="orderDetailTabs">
                <div class="tab active" data-tab="details">Details</div>
                <div class="tab" data-tab="milestones">Milestones</div>
                <div class="tab" data-tab="documents">Documents</div>
                <div class="tab" data-tab="lineItems">Line Items</div>
                <div class="tab" data-tab="messages">Messages</div>
                <div class="tab" data-tab="changeLog">Change Log</div>
            </div>

            <div id="orderDetailTabContent">
                <div class="tab-content active" id="tab-details">
                    <div class="grid-2" style="gap: 24px">
                        <div>
                            <h4 style="font-size: 14px; font-weight: 600; margin-bottom: 16px; color: var(--text)">Order Information</h4>
                            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 12px">
                                <div style="padding: 12px; background: var(--bg-light); border-radius: 6px">
                                    <div style="font-size: 11px; color: var(--text-muted); margin-bottom: 4px">Incoterm</div>
                                    <div style="font-weight: 600">${order.incoterm}</div>
                                </div>
                                <div style="padding: 12px; background: var(--bg-light); border-radius: 6px">
                                    <div style="font-size: 11px; color: var(--text-muted); margin-bottom: 4px">Mode</div>
                                    <div style="font-weight: 600">${order.mode}</div>
                                </div>
                                <div style="padding: 12px; background: var(--bg-light); border-radius: 6px">
                                    <div style="font-size: 11px; color: var(--text-muted); margin-bottom: 4px">Total Value</div>
                                    <div style="font-weight: 600">${ConexusHelpers.formatCurrency(order.totalValue)}</div>
                                </div>
                                <div style="padding: 12px; background: var(--bg-light); border-radius: 6px">
                                    <div style="font-size: 11px; color: var(--text-muted); margin-bottom: 4px">Order Date</div>
                                    <div style="font-weight: 600">${ConexusHelpers.formatDate(order.orderDate)}</div>
                                </div>
                            </div>

                            <h4 style="font-size: 14px; font-weight: 600; margin: 24px 0 16px; color: var(--text)">Shipment Details</h4>
                            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 12px">
                                <div style="padding: 12px; background: var(--bg-light); border-radius: 6px">
                                    <div style="font-size: 11px; color: var(--text-muted); margin-bottom: 4px">Container</div>
                                    <div style="font-weight: 600">${order.containerRef || '-'}</div>
                                </div>
                                <div style="padding: 12px; background: var(--bg-light); border-radius: 6px">
                                    <div style="font-size: 11px; color: var(--text-muted); margin-bottom: 4px">MBL / AWB</div>
                                    <div style="font-weight: 600">${order.mblNumber || '-'}</div>
                                </div>
                                <div style="padding: 12px; background: var(--bg-light); border-radius: 6px">
                                    <div style="font-size: 11px; color: var(--text-muted); margin-bottom: 4px">Vessel</div>
                                    <div style="font-weight: 600">${order.vesselName || '-'}</div>
                                </div>
                                <div style="padding: 12px; background: var(--bg-light); border-radius: 6px">
                                    <div style="font-size: 11px; color: var(--text-muted); margin-bottom: 4px">Voyage</div>
                                    <div style="font-weight: 600">${order.voyageNumber || '-'}</div>
                                </div>
                            </div>
                        </div>

                        <div>
                            <h4 style="font-size: 14px; font-weight: 600; margin-bottom: 16px; color: var(--text)">Parties</h4>
                            <div style="display: flex; flex-direction: column; gap: 12px">
                                <div style="padding: 12px; background: var(--bg-light); border-radius: 6px">
                                    <div style="font-size: 11px; color: var(--text-muted); margin-bottom: 4px">Buyer</div>
                                    <div style="font-weight: 600">${buyer?.name || '-'}</div>
                                    <div style="font-size: 12px; color: var(--text-muted)">${buyer?.contact || ''}</div>
                                </div>
                                <div style="padding: 12px; background: var(--bg-light); border-radius: 6px">
                                    <div style="font-size: 11px; color: var(--text-muted); margin-bottom: 4px">Supplier</div>
                                    <div style="font-weight: 600">${supplier?.name || '-'}</div>
                                    <div style="font-size: 12px; color: var(--text-muted)">${supplier?.city}, ${supplier?.country}</div>
                                </div>
                                <div style="padding: 12px; background: var(--bg-light); border-radius: 6px">
                                    <div style="font-size: 11px; color: var(--text-muted); margin-bottom: 4px">Forwarder</div>
                                    <div style="font-weight: 600">${forwarder?.name || '-'}</div>
                                </div>
                            </div>

                            <h4 style="font-size: 14px; font-weight: 600; margin: 24px 0 16px; color: var(--text)">Key Dates</h4>
                            <div style="display: flex; flex-direction: column; gap: 12px">
                                <div style="display: flex; justify-content: space-between; padding: 8px 12px; background: var(--bg-light); border-radius: 6px">
                                    <span style="font-size: 13px; color: var(--text-muted)">Cargo Ready</span>
                                    <span style="font-weight: 600">${ConexusHelpers.formatDate(order.cargoReadyDate)}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; padding: 8px 12px; background: var(--bg-light); border-radius: 6px">
                                    <span style="font-size: 13px; color: var(--text-muted)">ETD</span>
                                    <span style="font-weight: 600">${ConexusHelpers.formatDate(order.etd)}</span>
                                </div>
                                <div style="display: flex; justify-content: space-between; padding: 8px 12px; background: var(--accent-light); border-radius: 6px">
                                    <span style="font-size: 13px; color: var(--text-muted)">ETA</span>
                                    <span class="confidence">
                                        <span class="confidence-dot ${order.etaConfidence?.toLowerCase()}"></span>
                                        <strong>${ConexusHelpers.formatDate(order.eta)}</strong>
                                        <span style="font-size: 11px; color: var(--text-muted)">(${order.etaConfidence})</span>
                                    </span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="tab-content" id="tab-milestones">
                    <div class="timeline">
                        ${order.milestones.map(m => `
                            <div class="timeline-item">
                                <div class="timeline-marker ${m.completed ? 'completed' : 'pending'}">
                                    ${m.completed ? '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3"><polyline points="20 6 9 17 4 12"/></svg>' : ''}
                                </div>
                                <div class="timeline-content">
                                    <div class="timeline-header">
                                        <span class="timeline-title">${m.name}</span>
                                        <span class="timeline-date">${m.date ? ConexusHelpers.formatDate(m.date) : 'Pending'}</span>
                                    </div>
                                    ${m.note ? `<div class="timeline-note">${m.note}</div>` : ''}
                                </div>
                            </div>
                        `).join('')}
                    </div>
                </div>

                <div class="tab-content" id="tab-documents">
                    ${order.documents.length === 0 ? `
                        <div class="empty-state">
                            <svg class="empty-state-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                <polyline points="14 2 14 8 20 8"/>
                            </svg>
                            <h3 class="empty-state-title">No documents</h3>
                            <p class="empty-state-description">Upload documents to attach them to this order</p>
                            <button class="btn btn-primary" onclick="ConexusApp.uploadDocument('${order.id}')">Upload Document</button>
                        </div>
                    ` : `
                        <div style="display: flex; justify-content: flex-end; margin-bottom: 16px">
                            <button class="btn btn-primary btn-sm" onclick="ConexusApp.uploadDocument('${order.id}')">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="14" height="14">
                                    <line x1="12" y1="5" x2="12" y2="19"/><line x1="5" y1="12" x2="19" y2="12"/>
                                </svg>
                                Upload Document
                            </button>
                        </div>
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Document Type</th>
                                    <th>File Name</th>
                                    <th>Upload Date</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${order.documents.map(doc => `
                                    <tr>
                                        <td><strong>${doc.type}</strong></td>
                                        <td>${doc.fileName || '-'}</td>
                                        <td>${doc.uploadDate ? ConexusHelpers.formatDate(doc.uploadDate) : '-'}</td>
                                        <td>
                                            <span class="badge ${doc.status === 'Uploaded' ? 'badge-success' : 'badge-error'}">${doc.status}</span>
                                        </td>
                                        <td>
                                            ${doc.status === 'Uploaded' ? `
                                                <button class="btn btn-sm btn-ghost" onclick="ConexusApp.viewDocument('${doc.fileName}')">View</button>
                                                <button class="btn btn-sm btn-ghost" onclick="ConexusApp.downloadDocument('${doc.fileName}')">Download</button>
                                            ` : `
                                                <button class="btn btn-sm btn-primary" onclick="ConexusApp.uploadDocument('${order.id}', '${doc.type}')">Upload</button>
                                            `}
                                        </td>
                                    </tr>
                                `).join('')}
                            </tbody>
                        </table>
                    `}
                </div>

                <div class="tab-content" id="tab-lineItems">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>SKU</th>
                                <th>Description</th>
                                <th>Quantity</th>
                                <th>Unit</th>
                                <th>Unit Price</th>
                                <th>Total</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${order.lineItems.map(item => `
                                <tr>
                                    <td><strong>${item.sku}</strong></td>
                                    <td>${item.description}</td>
                                    <td>${item.quantity.toLocaleString()}</td>
                                    <td>${item.unit}</td>
                                    <td>${ConexusHelpers.formatCurrency(item.unitPrice)}</td>
                                    <td><strong>${ConexusHelpers.formatCurrency(item.totalPrice)}</strong></td>
                                </tr>
                            `).join('')}
                        </tbody>
                        <tfoot>
                            <tr>
                                <td colspan="5" style="text-align: right"><strong>Total Order Value:</strong></td>
                                <td><strong>${ConexusHelpers.formatCurrency(order.totalValue)}</strong></td>
                            </tr>
                        </tfoot>
                    </table>
                </div>

                <div class="tab-content" id="tab-messages">
                    ${order.messages.length === 0 ? `
                        <div class="empty-state">
                            <svg class="empty-state-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"/>
                            </svg>
                            <h3 class="empty-state-title">No messages</h3>
                            <p class="empty-state-description">Start a conversation about this order</p>
                        </div>
                    ` : order.messages.map(msg => `
                        <div style="padding: 12px; background: var(--bg-light); border-radius: 8px; margin-bottom: 12px">
                            <div style="display: flex; justify-content: space-between; margin-bottom: 8px">
                                <strong>${msg.from}</strong>
                                <span style="font-size: 12px; color: var(--text-muted)">${msg.date}</span>
                            </div>
                            <p style="font-size: 13px; color: var(--text-secondary)">${msg.message}</p>
                        </div>
                    `).join('')}
                    <div style="margin-top: 16px">
                        <textarea class="form-textarea" rows="3" placeholder="Type a message..." id="orderMessageInput"></textarea>
                        <div style="display: flex; justify-content: flex-end; margin-top: 8px">
                            <button class="btn btn-primary" onclick="ConexusApp.sendOrderMessage('${order.id}')">Send Message</button>
                        </div>
                    </div>
                </div>

                <div class="tab-content" id="tab-changeLog">
                    ${order.changeLog.length === 0 ? `
                        <div class="empty-state">
                            <svg class="empty-state-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                <polyline points="14 2 14 8 20 8"/>
                            </svg>
                            <h3 class="empty-state-title">No changes recorded</h3>
                            <p class="empty-state-description">Order changes will be logged here</p>
                        </div>
                    ` : `
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Date</th>
                                    <th>Field</th>
                                    <th>Old Value</th>
                                    <th>New Value</th>
                                    <th>Changed By</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${order.changeLog.map(change => `
                                    <tr>
                                        <td>${ConexusHelpers.formatDate(change.date)}</td>
                                        <td><strong>${change.field}</strong></td>
                                        <td><span style="color: var(--error)">${change.oldValue}</span></td>
                                        <td><span style="color: var(--success)">${change.newValue}</span></td>
                                        <td>${change.changedBy}</td>
                                    </tr>
                                `).join('')}
                            </tbody>
                        </table>
                    `}
                </div>
            </div>
        `;

        // Setup tabs
        content.querySelectorAll('.tab').forEach(tab => {
            tab.addEventListener('click', () => {
                content.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
                content.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));
                tab.classList.add('active');
                content.querySelector(`#tab-${tab.dataset.tab}`).classList.add('active');
            });
        });

        this.openModal('orderDetailModal');
    },

    // ==================== ACTIONS ====================
    filterByStatus(status) {
        this.state.filters.status = status;
        if (this.state.currentPage === 'dashboard') {
            this.renderPipeline();
            this.renderRecentOrders();
        } else if (this.state.currentPage === 'orders') {
            this.renderOrdersList();
        }
    },

    filterAlerts(filterType, value) {
        if (filterType === 'priority') {
            this.state.filters.alertPriority = value;
        } else if (filterType === 'type') {
            this.state.filters.alertType = value;
        }
        this.renderAlerts();
    },

    handleGlobalSearch(query) {
        this.state.searchQuery = query;
        // Could implement live search suggestions here
    },

    executeGlobalSearch(query) {
        if (!query.trim()) return;
        
        const results = ConexusHelpers.searchOrders(query);
        if (results.length === 1) {
            this.openOrderDetail(results[0].id);
        } else if (results.length > 1) {
            this.state.searchQuery = query;
            this.navigateTo('orders');
            this.renderOrdersList();
        } else {
            this.showToast('No orders found matching your search', 'info');
        }
    },

    filterOrderDesk(query) {
        const items = document.querySelectorAll('.order-desk-item');
        items.forEach(item => {
            const orderId = item.dataset.orderId;
            const order = ConexusHelpers.getOrder(orderId);
            const supplier = ConexusHelpers.getSupplier(order?.supplierId);
            const searchText = `${orderId} ${supplier?.name || ''}`.toLowerCase();
            item.style.display = searchText.includes(query.toLowerCase()) ? '' : 'none';
        });
    },

    handleAlertAction(alertId, action) {
        this.showToast(`Action "${action}" triggered for alert ${alertId}`, 'success');
        // In real implementation, would call API and update state
    },

    acknowledgeAlert(alertId) {
        const alert = ConexusData.alerts.find(a => a.id === alertId);
        if (alert) {
            if (alert.status === 'Acknowledged') {
                alert.status = 'Resolved';
                this.showToast('Alert resolved', 'success');
            } else {
                alert.status = 'Acknowledged';
                this.showToast('Alert acknowledged', 'success');
            }
            this.renderAlerts();
            this.updateAlertBadges();
        }
    },

    validateAllOrders() {
        this.showToast('Validating all orders...', 'info');
        // Simulate validation
        setTimeout(() => {
            this.showToast('Validation complete. 3 orders require manual review.', 'success');
        }, 1500);
    },

    completeOrderData(orderId) {
        this.openOrderDetail(orderId);
        this.showToast('Please complete the missing fields', 'info');
    },

    editOrder(orderId) {
        this.openOrderDetail(orderId);
    },

    messageOrder(orderId) {
        this.openOrderDetail(orderId);
        // Switch to messages tab
        setTimeout(() => {
            const messagesTab = document.querySelector('[data-tab="messages"]');
            if (messagesTab) messagesTab.click();
        }, 100);
    },

    sendOrderMessage(orderId) {
        const input = document.getElementById('orderMessageInput');
        if (!input || !input.value.trim()) return;

        const order = ConexusHelpers.getOrder(orderId);
        if (order) {
            order.messages.push({
                from: ConexusData.user.name,
                date: new Date().toLocaleString(),
                message: input.value.trim()
            });
            this.openOrderDetail(orderId);
            setTimeout(() => {
                const messagesTab = document.querySelector('[data-tab="messages"]');
                if (messagesTab) messagesTab.click();
            }, 100);
            this.showToast('Message sent', 'success');
        }
    },

    trackVessel(orderId) {
        const order = ConexusHelpers.getOrder(orderId);
        if (order && order.vesselName) {
            this.showToast(`Opening tracking for ${order.vesselName}...`, 'info');
            // In real implementation, would open vessel tracking
        }
    },

    uploadDocument(orderId, docType) {
        this.showToast('Document upload dialog would open here', 'info');
    },

    viewDocument(fileName) {
        this.showToast(`Opening ${fileName}...`, 'info');
    },

    downloadDocument(fileName) {
        this.showToast(`Downloading ${fileName}...`, 'info');
    },

    viewSupplierForecast(supplierId) {
        const supplier = ConexusHelpers.getSupplier(supplierId);
        this.showToast(`Opening forecast for ${supplier?.name}`, 'info');
    },

    exportCO2Report() {
        this.showToast('Generating CO₂ report...', 'info');
        setTimeout(() => {
            this.showToast('CO₂ report downloaded', 'success');
        }, 1500);
    },

    openAlertDetail(alertId) {
        const alert = ConexusData.alerts.find(a => a.id === alertId);
        if (!alert) return;
        
        // Could open a detailed alert modal
        this.showToast(`Alert: ${alert.title}`, 'info');
    },

    // ==================== NOTIFICATIONS ====================
    loadNotifications() {
        this.state.notifications = ConexusData.notifications;
    },

    updateAlertBadges() {
        const criticalCount = ConexusHelpers.getCriticalAlertsCount();
        const alertBadges = document.querySelectorAll('.nav-badge.critical');
        alertBadges.forEach(badge => {
            badge.textContent = criticalCount;
            badge.style.display = criticalCount > 0 ? '' : 'none';
        });
    },

    toggleNotifications() {
        this.showToast('Notifications panel would open here', 'info');
    },

    // ==================== UTILITIES ====================
    showToast(message, type = 'info') {
        // Create toast element
        const toast = document.createElement('div');
        toast.style.cssText = `
            position: fixed;
            bottom: 20px;
            right: 20px;
            padding: 12px 20px;
            background: ${type === 'success' ? 'var(--success)' : type === 'error' ? 'var(--error)' : 'var(--primary)'};
            color: white;
            border-radius: 8px;
            box-shadow: var(--shadow-lg);
            z-index: 10000;
            animation: slideIn 0.3s ease;
            font-size: 13px;
            font-weight: 500;
        `;
        toast.textContent = message;
        document.body.appendChild(toast);

        // Remove after 3 seconds
        setTimeout(() => {
            toast.style.animation = 'slideOut 0.3s ease';
            setTimeout(() => toast.remove(), 300);
        }, 3000);
    }
};

// Add toast animations
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
`;
document.head.appendChild(style);

// Initialize app when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    ConexusApp.init();
});
